<?php

class AuthController extends Controller
{
    public function login(): void
    {
        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $studentId = trim($_POST['student_id'] ?? '');
            $password  = $_POST['password'] ?? '';

            $userModel = new User();
            $user      = $userModel->findByStudentId($studentId);

            if ($user && password_verify($password, $user['password_hash'])) {
                if (($user['status'] ?? 'active') !== 'active') {
                    $error = 'Your account is suspended. Contact PACCPOLICY.';
                } else {
                    session_regenerate_id(true);
                    $_SESSION['user'] = [
                        'id'             => $user['id'],
                        'student_id'     => $user['student_id'],
                        'role'           => $user['role'],
                        'name'           => $user['full_name'],
                        'profile_image'  => $user['profile_image'],
                    ];
                    $this->redirect('/dashboard');
                    return;
                }
            } else {
                $error = 'Invalid Student ID or password.';
            }

            $this->view('auth/login', [
                'title' => 'Login',
                'error' => $error ?? null,
            ]);
            return;
        }

        $this->view('auth/login', [
            'title' => 'Login',
        ]);
    }

    public function register(): void
    {
        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $studentId  = trim($_POST['student_id'] ?? '');
            $fullName   = trim($_POST['full_name'] ?? '');
            $institution = trim($_POST['institution'] ?? '');
            $programme  = trim($_POST['programme'] ?? '');
            $password   = $_POST['password'] ?? '';
            $confirm    = $_POST['password_confirm'] ?? '';

            $errors = [];
            if ($studentId === '' || $fullName === '' || $institution === '' || $programme === '') {
                $errors[] = 'All fields are required.';
            }
            if (strlen($password) < 8) {
                $errors[] = 'Password must be at least 8 characters.';
            }
            if ($password !== $confirm) {
                $errors[] = 'Passwords do not match.';
            }

            $userModel = new User();
            if ($userModel->findByStudentId($studentId)) {
                $errors[] = 'An account with this Student ID already exists.';
            }

            if (empty($errors)) {
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $userModel->createStudent($studentId, $fullName, $institution, $programme, $hash);
                $this->redirect('/auth/login');
                return;
            }

            $this->view('auth/register', [
                'title'  => 'Register',
                'errors' => $errors,
            ]);
            return;
        }

        $this->view('auth/register', [
            'title' => 'Register',
        ]);
    }

    public function logout(): void
    {
        $_SESSION = [];
        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params['path'], $params['domain'],
                $params['secure'], $params['httponly']
            );
        }
        session_destroy();
        $this->redirect('/auth/login');
    }

    public function forgotPassword(): void
    {
        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $studentId = trim($_POST['student_id'] ?? '');
            $errors = [];
            $success = null;

            $userModel = new User();
            $user = $userModel->findByStudentId($studentId);

            if (!$user) {
                $errors[] = 'No account found with this Student ID.';
            } else {
                // Create password reset token
                $token = $userModel->createPasswordResetToken($user['id']);
                $resetLink = BASE_URL . '/auth/resetPassword/' . $token;
                
                // In a real application, you would send an email here
                // For now, we'll store the link in the session for testing
                $_SESSION['reset_link'] = $resetLink;
                $_SESSION['reset_email'] = $user['email'] ?? 'not-set';
                
                $success = 'Password reset link has been created. Check your email or note the reset link.';
            }

            $this->view('auth/forgotPassword', [
                'title'   => 'Forgot Password',
                'errors'  => $errors,
                'success' => $success,
            ]);
            return;
        }

        $this->view('auth/forgotPassword', [
            'title' => 'Forgot Password',
        ]);
    }

    public function resetPassword(string $token = ''): void
    {
        if (empty($token)) {
            $this->view('auth/forgotPassword', [
                'title' => 'Reset Password',
                'error' => 'Invalid reset link.',
            ]);
            return;
        }

        if ($this->isPost()) {
            if (!$this->verifyCsrf()) {
                http_response_code(400);
                echo 'Invalid CSRF token';
                return;
            }

            $newPassword = $_POST['password'] ?? '';
            $confirmPassword = $_POST['confirm_password'] ?? '';

            $errors = [];

            if (strlen($newPassword) < 8) {
                $errors[] = 'Password must be at least 8 characters.';
            }

            if ($newPassword !== $confirmPassword) {
                $errors[] = 'Passwords do not match.';
            }

            if (empty($errors)) {
                $userModel = new User();
                if ($userModel->resetPassword($token, password_hash($newPassword, PASSWORD_BCRYPT))) {
                    $this->view('auth/resetPasswordSuccess', [
                        'title' => 'Password Reset',
                    ]);
                    return;
                } else {
                    $errors[] = 'Invalid or expired reset link.';
                }
            }

            $this->view('auth/resetPassword', [
                'title'  => 'Reset Password',
                'token'  => $token,
                'errors' => $errors,
            ]);
            return;
        }

        // Validate token
        $userModel = new User();
        $tokenData = $userModel->validatePasswordResetToken($token);

        if (!$tokenData) {
            $this->view('auth/forgotPassword', [
                'title' => 'Reset Password',
                'error' => 'Invalid or expired reset link.',
            ]);
            return;
        }

        $this->view('auth/resetPassword', [
            'title' => 'Reset Password',
            'token' => $token,
        ]);
    }
}

